// Configuração do cliente Supabase
const supabaseUrl = "https://fdoerrjxisyauuiginea.supabase.co";
const supabaseKey =
  "eyJhbGciOiJIUzI1NiIsInR5cCI6IkpXVCJ9.eyJpc3MiOiJzdXBhYmFzZSIsInJlZiI6ImZkb2Vycmp4aXN5YXV1aWdpbmVhIiwicm9sZSI6ImFub24iLCJpYXQiOjE3NDc3MDYxMDIsImV4cCI6MjA2MzI4MjEwMn0.pEvyNhYJgqcCOgnNTJwo-Yk0easUjirsltVXBkdkZ_A";
// Inicializa o cliente do Supabase (certifique-se de que a biblioteca foi carregada)
const supabase = window.supabase.createClient(supabaseUrl, supabaseKey);

// Status de autenticação
let currentUser = null;

// Configurações para login com Google
const googleRedirectURL = window.location.origin + "/Pate-papo-com-IA/login.html";

// Evento para detectar mudanças de autenticação
supabase.auth.onAuthStateChange((event, session) => {
  if (event === 'SIGNED_IN' && session) {
    currentUser = session.user;
    console.log('Usuário autenticado via evento:', currentUser.email);
  } else if (event === 'SIGNED_OUT') {
    currentUser = null;
    console.log('Usuário desconectado');
  }
});

// Verificar se há um usuário já logado
async function checkSession() {
  try {
    const { data, error } = await supabase.auth.getSession();

    if (error) {
      console.error("Erro ao verificar sessão:", error);
      return null;
    }

    if (data && data.session) {
      currentUser = data.session.user;
      return currentUser;
    }

    return null;
  } catch (err) {
    console.error("Erro ao verificar sessão:", err);
    return null;
  }
}

// Login com e-mail e senha
async function loginWithEmail(email, password) {
  try {
    const { data, error } = await supabase.auth.signInWithPassword({
      email: email,
      password: password,
    });

    if (error) {
      throw error;
    }

    currentUser = data.user;
    return { success: true, user: currentUser };
  } catch (error) {
    console.error("Erro de login:", error);
    return {
      success: false,
      error: error.message || "Falha no login. Verifique suas credenciais.",
    };
  }
}

// Registrar novo usuário
async function registerWithEmail(email, password) {
  try {
    const { data, error } = await supabase.auth.signUp({
      email: email,
      password: password,
    });

    if (error) {
      throw error;
    }

    // Verificar se precisa de confirmação por e-mail
    if (
      data.user &&
      data.user.identities &&
      data.user.identities.length === 0
    ) {
      return {
        success: false,
        error:
          "Este e-mail já está registrado. Faça login ou recupere sua senha.",
      };
    }

    currentUser = data.user;
    return {
      success: true,
      user: currentUser,
      message: data.session
        ? "Registro concluído com sucesso!"
        : "Confira seu e-mail para verificar sua conta.",
    };
  } catch (error) {
    console.error("Erro no registro:", error);
    return {
      success: false,
      error: error.message || "Falha no registro. Tente novamente.",
    };
  }
}

// Recuperação de senha
async function resetPassword(email) {
  try {
    const { error } = await supabase.auth.resetPasswordForEmail(email, {
      redirectTo: window.location.origin + "/Pate-papo-com-IA/reset-password.html",
    });

    if (error) {
      throw error;
    }

    return {
      success: true,
      message: "Instruções de recuperação enviadas para seu e-mail.",
    };
  } catch (error) {
    console.error("Erro na recuperação de senha:", error);
    return {
      success: false,
      error: error.message || "Falha ao enviar instruções de recuperação.",
    };
  }
}

// Login com Google
async function loginWithGoogle() {
  try {
    const { data, error } = await supabase.auth.signInWithOAuth({
      provider: 'google',
      options: {
        redirectTo: googleRedirectURL,
        scopes: 'email profile',
        queryParams: {
          access_type: 'offline',
          prompt: 'consent'
        }
      }
    });
    
    if (error) {
      throw error;
    }
    
    return { success: true };
  } catch (error) {
    console.error("Erro no login com Google:", error);
    return { 
      success: false, 
      error: error.message || "Falha no login com Google."
    };
  }
}

// Verificar resposta do login com OAuth após redirecionamento
async function handleOAuthResponse() {
  try {
    // Processar hash da URL para extrair credenciais
    const hashParams = new URLSearchParams(window.location.hash.substring(1));
    const accessToken = hashParams.get('access_token');
    
    // Se temos um token na URL, tentar processar diretamente
    if (accessToken) {
      console.log('Token encontrado na URL, processando...');
      try {
        const { data: userData, error: userError } = await supabase.auth.getUser(accessToken);
        if (userError) throw userError;
        if (userData && userData.user) {
          currentUser = userData.user;
          return { success: true, user: currentUser };
        }
      } catch (tokenError) {
        console.warn('Erro ao processar token da URL:', tokenError);
        // Continuar com getSession() se falhar
      }
    }
    
    // Método padrão usando getSession
    const { data, error } = await supabase.auth.getSession();
    
    if (error) {
      throw error;
    }
    
    if (data && data.session) {
      currentUser = data.session.user;
      // Garantir que qualquer informação de hash na URL seja limpa após processamento
      window.history.replaceState(null, document.title, window.location.pathname + window.location.search);
      return { success: true, user: currentUser };
    }
    
    return { success: false, error: "Nenhuma sessão encontrada" };
  } catch (error) {
    console.error("Erro ao processar resposta OAuth:", error);
    return { 
      success: false, 
      error: error.message || "Falha ao processar autenticação."
    };
  }
}

// Logout
async function logout() {
  try {
    const { error } = await supabase.auth.signOut();

    if (error) {
      throw error;
    }

    currentUser = null;
    return { success: true };
  } catch (error) {
    console.error("Erro no logout:", error);
    return {
      success: false,
      error: error.message || "Falha ao fazer logout.",
    };
  }
}

// Função para atualizar o webhook URL para o usuário atual
async function updateUserWebhook(webhookUrl) {
  // Verificar se o usuário está autenticado
  if (!currentUser) {
    return {
      success: false,
      error: "Usuário não autenticado",
    };
  }

  try {
    const { data, error } = await supabase.from("user_preferences").upsert(
      {
        user_id: currentUser.id,
        webhook_url: webhookUrl,
        updated_at: new Date(),
      },
      {
        onConflict: "user_id",
      },
    );

    if (error) {
      throw error;
    }

    return {
      success: true,
      message: "Webhook atualizado com sucesso",
    };
  } catch (error) {
    console.error("Erro ao atualizar webhook:", error);
    return {
      success: false,
      error: error.message || "Falha ao atualizar webhook.",
    };
  }
}

// Função para obter as configurações do usuário atual
async function getUserSettings() {
  // Verificar se o usuário está autenticado
  if (!currentUser) {
    // Tentar atualizar o estado do usuário antes de retornar erro
    try {
      const { data } = await supabase.auth.getSession();
      if (data && data.session) {
        currentUser = data.session.user;
      } else {
        return {
          success: false,
          error: "Usuário não autenticado",
        };
      }
    } catch (sessionError) {
      console.error("Erro ao verificar sessão:", sessionError);
      return {
        success: false,
        error: "Usuário não autenticado",
      };
    }
  }
  
  try {
    const { data, error } = await supabase
      .from("user_preferences")
      .select("*")
      .eq("user_id", currentUser.id)
      .single();

    if (error && error.code !== "PGRST116") {
      // PGRST116 é o erro de nenhum resultado encontrado
      throw error;
    }

    return {
      success: true,
      settings: data || {},
    };
  } catch (error) {
    console.error("Erro ao obter configurações do usuário:", error);
    return {
      success: false,
      error: error.message || "Falha ao obter configurações.",
    };
  }
}
