(function () {
  const chatInput = document.getElementById("chat-input");
  const sendBtn = document.getElementById("send-btn");
  const chatMessages = document.getElementById("chat-messages");
  const typingIndicator = document.getElementById("typing-indicator");
  const clearHistoryBtn = document.getElementById("clear-history");
  const toggleThemeBtn = document.getElementById("toggle-theme");
  const clearModal = document.getElementById("clear-modal");
  const cancelClearBtn = document.getElementById("cancel-clear");
  const confirmClearBtn = document.getElementById("confirm-clear");
  const settingsBtn = document.getElementById("settings-btn");
  const settingsModal = document.getElementById("settings-modal");
  const webhookUrlInput = document.getElementById("webhook-url");
  const cancelSettingsBtn = document.getElementById("cancel-settings");
  const saveSettingsBtn = document.getElementById("save-settings");

  let isWaitingForResponse = false;
  let chatHistory = [];
  let webhookUrl = "https://n8n.ichatbot.com.br/webhook/chat-com-ia";
  let userSettings = null;

  const storage = {
    save: function (key, value) {
      try {
        localStorage.setItem(key, JSON.stringify(value));
        return true;
      } catch (e) {
        this["_" + key] = value;
        return false;
      }
    },
    load: function (key) {
      try {
        const item = localStorage.getItem(key);
        return item ? JSON.parse(item) : null;
      } catch (e) {
        return this["_" + key] || null;
      }
    },
    remove: function (key) {
      try {
        localStorage.removeItem(key);
        return true;
      } catch (e) {
        delete this["_" + key];
        return false;
      }
    },
  };

  function formatBotMessage(text) {
    if (!text) return "";
    
    // Verificar se o texto é um objeto JSON e extrair o conteúdo se necessário
    try {
      const jsonObj = JSON.parse(text);
      if (jsonObj && jsonObj.output) {
        text = jsonObj.output;
      }
    } catch (e) {
      // Não é JSON válido, continua com o texto original
    }
    
    // Tratar literais de quebra de linha (strings \n e \n\n) para quebras de linha HTML
    text = text.replace(/\\n\\n/g, "\n\n");
    text = text.replace(/\\n/g, "\n");
    
    // Tratamento mais completo para formatação de e-mails
    // 1. Remover caracteres de citação de email (>) no início das linhas
    text = text.replace(/^>\s*/gm, "");
    text = text.replace(/\n>\s*/g, "\n");
    
    // 2. Tratar links de botões comuns em e-mails
    text = text.replace(/\[([^\]]+)\]\(([^)]+)\)/g, '<a href="$2" target="_blank" rel="noopener noreferrer">$1</a>');
    
    // 3. Tratar cabeçalhos de e-mail (De:, Para:, Assunto:, etc.)
    text = text.replace(/^(From|De|Para|To|Assunto|Subject|Date|Data):\s*(.*?)$/gm, '<strong>$1:</strong> $2');
    
    let escaped = text.replace(/&/g, "&amp;").replace(/</g, "&lt;").replace(/>/g, "&gt;");
    
    // Remover URLs de acompanhamento comuns em e-mails
    escaped = escaped.replace(/https?:\/\/[^\s]+?(?:click|track|unsubscribe|redirect)[^\s]*?/g, function(match) {
      return `<span class="email-link">${match}</span>`;
    });
    
    // Headings
    escaped = escaped.replace(/###\s+(.*?)$/gm, "<h3>$1</h3>");
    escaped = escaped.replace(/##\s+(.*?)$/gm, "<h2>$1</h2>");
    escaped = escaped.replace(/#\s+(.*?)$/gm, "<h1>$1</h1>");
    
    // Bold and italic
    escaped = escaped.replace(/\*\*(.*?)\*\*/g, "<strong>$1</strong>");
    escaped = escaped.replace(/\*(.*?)\*/g, "<em>$1</em>");
    
    // Horizontal rule
    escaped = escaped.replace(/^---+$/gm, "<hr>");
    
    // Tables - Process markdown tables - fixed implementation
    escaped = escaped.replace(/(\|[^\n]+\|\r?\n)((?:\|[-:]+[-:|\s]*\|\r?\n))((?:\|[^\n]+\|\r?\n)+)/g, function(match, headerPart, separatorPart, bodyPart) {
      let tableHTML = '<div class="table-wrapper"><table class="excel-table">';
      
      // Process header row
      const headerRow = headerPart.trim();
      const headers = headerRow.split('|').slice(1, -1).map(cell => cell.trim());
      
      tableHTML += '<thead><tr>';
      headers.forEach(header => {
        tableHTML += `<th>${header}</th>`;
      });
      tableHTML += '</tr></thead>';
      
      // Process body rows
      tableHTML += '<tbody>';
      
      const bodyRows = bodyPart.trim().split('\n');
      bodyRows.forEach(row => {
        if (!row.includes('|')) return;
        
        // Split by pipe and remove empty first/last elements
        const cells = row.split('|').slice(1, -1).map(cell => cell.trim());
        
        tableHTML += '<tr>';
        cells.forEach(cell => {
          // Handle special formatting inside cells (bold, italics, etc.)
          cell = cell.replace(/\*\*(.*?)\*\*/g, "<strong>$1</strong>");
          cell = cell.replace(/\*(.*?)\*/g, "<em>$1</em>");
          tableHTML += `<td>${cell}</td>`;
        });
        tableHTML += '</tr>';
      });
      
      tableHTML += '</tbody></table></div>';
      return tableHTML;
    });
    
    // Catch tables that might not have been formatted exactly according to spec
    escaped = escaped.replace(/\n(\|[^\n]+\|\r?\n\|[-:]+[-:|\s]*\|\r?\n)(\|[^\n]+\|[\s\S]*?)(?=\n\s*\n|$)/g, function(match, headerAndSeparator, bodyPart) {
      if (match.includes('<table')) return match; // Skip if already processed
      
      let tableHTML = '<div class="table-wrapper"><table class="excel-table">';
      
      // Extract and process header
      const headerRow = headerAndSeparator.split('\n')[0].trim();
      const headers = headerRow.split('|').slice(1, -1).map(cell => cell.trim());
      
      tableHTML += '<thead><tr>';
      headers.forEach(header => {
        tableHTML += `<th>${header}</th>`;
      });
      tableHTML += '</tr></thead><tbody>';
      
      // Process body
      const bodyRows = bodyPart.trim().split('\n');
      bodyRows.forEach(row => {
        if (!row.includes('|')) return;
        
        const cells = row.split('|').slice(1, -1).map(cell => cell.trim());
        
        tableHTML += '<tr>';
        cells.forEach(cell => {
          cell = cell.replace(/\*\*(.*?)\*\*/g, "<strong>$1</strong>");
          cell = cell.replace(/\*(.*?)\*/g, "<em>$1</em>");
          tableHTML += `<td>${cell}</td>`;
        });
        tableHTML += '</tr>';
      });
      
      tableHTML += '</tbody></table></div>';
      return tableHTML;
    });
    
    // Lists - handling bullet points with and without indentation
    escaped = escaped.replace(/^\s*[-•]\s+(.*?)$/gm, "<li>$1</li>");
    escaped = escaped.replace(/(<li>.*?<\/li>\n)+/g, "<ul>$&</ul>");
    
    // Fix nested lists
    escaped = escaped.replace(/<\/ul>\n<ul>/g, "");
    
    // Melhorar detecção e formatação de padrões comuns em e-mails
    escaped = escaped.replace(/(?:(?:https?:\/\/)?(?:www\.)?)?[\w-]+(?:\.[\w-]+)+\.[a-z]{2,}(?:\/[^\s]*)?/g, function(match) {
        if (!match.startsWith('http')) {
            return `<a href="https://${match}" target="_blank" rel="noopener noreferrer">${match}</a>`;
        }
        return `<a href="${match}" target="_blank" rel="noopener noreferrer">${match}</a>`;
    });
    
    // Tratar botões de e-mail com formatação especial (ATUALIZAR, CANCELAR, etc. em maiúsculas)
    escaped = escaped.replace(/\b([A-Z]{5,}(?:\s+[A-Z]+)*)\b/g, '<span class="email-button">$1</span>');
    
    // Code blocks
    escaped = escaped.replace(/```(.*?)```/gs, "<pre><code>$1</code></pre>");
    escaped = escaped.replace(/`([^`]+)`/g, "<code>$1</code>");
    
    // Emoji support (preserving emojis)
    escaped = escaped.replace(/:([\w-]+):/g, function(match, emoji) {
      return match; // Keep emojis as is
    });
    
    // Paragraphs and line breaks - treated specifically for double and single line breaks
    // We do this after table processing to avoid breaking table structure
    escaped = escaped.replace(/\n\n+/g, "</p><p>");
    escaped = escaped.replace(/\n/g, "<br>");
    
    // Wrap in paragraph if not already wrapped
    if (!escaped.startsWith("<") || !escaped.match(/<(h[1-6]|ul|ol|pre|hr)/)) {
      escaped = "<p>" + escaped + "</p>";
    }
    
    // Clean up any empty paragraphs
    escaped = escaped.replace(/<p>\s*<\/p>/g, "");
    
    return escaped;
  }

  loadTheme();
  loadChatHistory();
  loadWebhookUrl();
  loadUserSettings();

  chatInput.addEventListener("input", function () {
    sendBtn.disabled = chatInput.value.trim() === "" || isWaitingForResponse;
  });

  chatInput.addEventListener("keypress", function (e) {
    if (
      e.key === "Enter" &&
      chatInput.value.trim() !== "" &&
      !isWaitingForResponse
    ) {
      sendMessage();
    }
  });

  sendBtn.addEventListener("click", function () {
    if (chatInput.value.trim() !== "" && !isWaitingForResponse) {
      sendMessage();
    }
  });

  clearHistoryBtn.addEventListener("click", function () {
    clearModal.style.display = "flex";
  });

  cancelClearBtn.addEventListener("click", function () {
    clearModal.style.display = "none";
  });

  confirmClearBtn.addEventListener("click", function () {
    clearChatHistory();
    clearModal.style.display = "none";
  });

  toggleThemeBtn.addEventListener("click", toggleTheme);

  settingsBtn.addEventListener("click", function() {
    webhookUrlInput.value = webhookUrl;
    settingsModal.style.display = "flex";
  });

  cancelSettingsBtn.addEventListener("click", function() {
    settingsModal.style.display = "none";
  });

  saveSettingsBtn.addEventListener("click", async function() {
    const newWebhookUrl = webhookUrlInput.value.trim();
    if (newWebhookUrl) {
      const settingsMessage = document.getElementById("settings-message");
      
      try {
        // Salvar no Supabase se o usuário estiver logado
        const user = await checkSession();
        if (user) {
          const result = await updateUserWebhook(newWebhookUrl);
          if (result.success) {
            webhookUrl = newWebhookUrl;
            settingsModal.style.display = "none";
          } else {
            settingsMessage.textContent = result.error;
            settingsMessage.className = "auth-message error";
            return;
          }
        } else {
          // Fallback para armazenamento local
          webhookUrl = newWebhookUrl;
          storage.save("webhookUrl", webhookUrl);
          settingsModal.style.display = "none";
        }
      } catch (error) {
        console.error("Erro ao salvar webhook:", error);
        // Fallback para armazenamento local em caso de erro
        webhookUrl = newWebhookUrl;
        storage.save("webhookUrl", webhookUrl);
        settingsModal.style.display = "none";
      }
    } else {
      const settingsMessage = document.getElementById("settings-message");
      settingsMessage.textContent = "Por favor, insira uma URL de webhook válida.";
      settingsMessage.className = "auth-message error";
    }
  });

  function sendMessage() {
    const message = chatInput.value.trim();
    // Capturar o timestamp quando a mensagem é enviada
    const now = new Date();
    const hours = now.getHours().toString().padStart(2, '0');
    const minutes = now.getMinutes().toString().padStart(2, '0');
    const timestamp = `${hours}:${minutes}`;
    
    appendMessage(message, "user", timestamp);
    chatInput.value = "";
    sendBtn.disabled = true;
    isWaitingForResponse = true;
    typingIndicator.style.display = "flex";
    scrollToBottom();

    fetch(webhookUrl, {
      method: "POST",
      headers: { "Content-Type": "application/json" },
      body: JSON.stringify({ message: message, history: chatHistory }),
    })
      .then((response) => {
        if (!response.ok)
          throw new Error(
            "Erro ao comunicar com o servidor: " + response.status,
          );
        return response.json();
      })
      .then((data) => {
        let responseText = "";
        if (typeof data === "string") {
          responseText = data;
        } else if (data.response && typeof data.response === "string") {
          responseText = data.response;
        } else if (data.message && typeof data.message === "string") {
          responseText = data.message;
        } else if (data.content && typeof data.content === "string") {
          responseText = data.content;
        } else {
          try {
            responseText = JSON.stringify(data);
          } catch (e) {
            responseText =
              "Recebi uma resposta, mas não consegui interpretá-la.";
          }
        }

        // Remover o processamento manual de JSON, já que isso agora é feito em formatBotMessage
        if (responseText.startsWith('{"output":"')) {
          try {
            const jsonObj = JSON.parse(responseText);
            if (jsonObj.output) {
              responseText = jsonObj.output;
            }
          } catch (e) {
            // Em caso de falha ao fazer parse, manter o texto original
          }
        }

        const typingDelay = Math.min(responseText.length * 10, 2000);
        setTimeout(() => {
          typingIndicator.style.display = "none";
          // Usar o mesmo timestamp para manter consistência
          const now = new Date();
          const hours = now.getHours().toString().padStart(2, '0');
          const minutes = now.getMinutes().toString().padStart(2, '0');
          const timestamp = `${hours}:${minutes}`;
          
          appendMessage(responseText, "ai", timestamp);
          updateChatHistory(message, responseText);
          isWaitingForResponse = false;
        }, typingDelay);
      })
      .catch((error) => {
        console.error("Erro:", error);
        typingIndicator.style.display = "none";
        appendMessage(
          "Desculpe, ocorreu um erro ao processar sua solicitação. Por favor, tente novamente.",
          "ai",
        );
        isWaitingForResponse = false;
      });
  }

  function appendMessage(text, type, timestamp = null) {
      const messageDiv = document.createElement("div");
      messageDiv.classList.add(
        "message",
        type === "user" ? "user-message" : "ai-message",
      );

      const iconDiv = document.createElement("div");
      iconDiv.classList.add(
        "message-icon",
        type === "user" ? "user-icon" : "bot-icon",
      );

      if (type === "user") {
        iconDiv.innerHTML = `
                  <svg xmlns="http://www.w3.org/2000/svg" width="16" height="16" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2" stroke-linecap="round" stroke-linejoin="round">
                      <path d="M20 21v-2a4 4 0 0 0-4-4H8a4 4 0 0 0-4 4v2"></path>
                      <circle cx="12" cy="7" r="4"></circle>
                  </svg>`;
      } else {
        iconDiv.innerHTML = `
                  <svg xmlns="http://www.w3.org/2000/svg" width="16" height="16" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2" stroke-linecap="round" stroke-linejoin="round">
                      <rect x="3" y="4" width="18" height="16" rx="2" ry="2"></rect>
                      <circle cx="9" cy="9" r="1"></circle>
                      <circle cx="15" cy="9" r="1"></circle>
                      <path d="M7 15h10"></path>
                  </svg>`;
      }

      const contentDiv = document.createElement("div");
      contentDiv.classList.add("message-content");
      if (type === "ai") {
        contentDiv.innerHTML = formatBotMessage(text);
      } else {
        contentDiv.textContent = text;
      }
    
      // Adicionar timestamp à mensagem
      const timestampDiv = document.createElement("div");
      timestampDiv.classList.add("message-timestamp");
      
      // Usar timestamp fornecido ou gerar um novo
      if (!timestamp) {
        const now = new Date();
        const hours = now.getHours().toString().padStart(2, '0');
        const minutes = now.getMinutes().toString().padStart(2, '0');
        timestamp = `${hours}:${minutes}`;
      }
      timestampDiv.textContent = timestamp;

      messageDiv.appendChild(iconDiv);
      messageDiv.appendChild(contentDiv);
      contentDiv.appendChild(timestampDiv);
      chatMessages.appendChild(messageDiv);
      scrollToBottom();
    }

  function scrollToBottom() {
    chatMessages.scrollTop = chatMessages.scrollHeight;
  }

  function updateChatHistory(userMessage, aiResponse) {
    // Capturar o timestamp atual para ambas as mensagens
    const now = new Date();
    const hours = now.getHours().toString().padStart(2, '0');
    const minutes = now.getMinutes().toString().padStart(2, '0');
    const timestamp = `${hours}:${minutes}`;
    
    chatHistory.push({ role: "user", content: userMessage, timestamp: timestamp });
    chatHistory.push({ role: "ai", content: aiResponse, timestamp: timestamp });
    if (chatHistory.length > 50) chatHistory = chatHistory.slice(-50);
    storage.save("chatHistory", chatHistory);
  }

  function loadChatHistory() {
    const savedHistory = storage.load("chatHistory");
    if (
      savedHistory &&
      Array.isArray(savedHistory) &&
      savedHistory.length > 0
    ) {
      chatHistory = savedHistory;
      chatMessages.innerHTML = "";
      
      // Primeira mensagem de boas-vindas
      if (chatHistory.length === 0 || chatHistory[0].role !== "ai") {
        // Adicionar timestamp à mensagem inicial
        const initialMsgTime = new Date();
        const initialHours = initialMsgTime.getHours().toString().padStart(2, '0');
        const initialMinutes = initialMsgTime.getMinutes().toString().padStart(2, '0');
        const initialTimestamp = `${initialHours}:${initialMinutes}`;
        appendMessage("Olá! Como posso ajudar você hoje?", "ai", initialTimestamp);
      }
      
      chatHistory.forEach((msg) => {
        if (msg && msg.content && typeof msg.content === "string") {
          appendMessage(msg.content, msg.role, msg.timestamp || null);
        }
      });
    }
  }

  function clearChatHistory() {
    chatHistory = [];
    storage.remove("chatHistory");
    chatMessages.innerHTML = "";
    // Adicionar timestamp à mensagem de boas-vindas após limpar histórico
    const initialMsgTime = new Date();
    const initialHours = initialMsgTime.getHours().toString().padStart(2, '0');
    const initialMinutes = initialMsgTime.getMinutes().toString().padStart(2, '0');
    const initialTimestamp = `${initialHours}:${initialMinutes}`;
    appendMessage("Olá! Como posso ajudar você hoje?", "ai", initialTimestamp);
  }

  function toggleTheme() {
    const isDark = document.body.getAttribute("data-theme") === "dark";
    const newTheme = isDark ? "light" : "dark";
    document.body.setAttribute("data-theme", newTheme);
    storage.save("theme", newTheme);
    updateThemeIcon(newTheme);
  }

  function loadTheme() {
    const savedTheme = storage.load("theme") || "light";
    document.body.setAttribute("data-theme", savedTheme);
    updateThemeIcon(savedTheme);
  }

  async function loadWebhookUrl() {
    // Primeiro tenta carregar do Supabase
    try {
      const user = await checkSession();
      if (user) {
        const result = await getUserSettings();
        if (result.success && result.settings && result.settings.webhook_url) {
          webhookUrl = result.settings.webhook_url;
          userSettings = result.settings;
          return;
        }
      }
    } catch (error) {
      console.error("Erro ao carregar webhook do Supabase:", error);
    }
    
    // Fallback para armazenamento local
    const savedWebhookUrl = storage.load("webhookUrl");
    if (savedWebhookUrl) {
      webhookUrl = savedWebhookUrl;
    }
  }
  
  async function loadUserSettings() {
    try {
      const user = await checkSession();
      if (user) {
        const result = await getUserSettings();
        if (result.success) {
          userSettings = result.settings;
        }
      }
    } catch (error) {
      console.error("Erro ao carregar configurações do usuário:", error);
    }
  }

  function updateThemeIcon(theme) {
    if (theme === "dark") {
      toggleThemeBtn.innerHTML = `
                <svg xmlns="http://www.w3.org/2000/svg" width="20" height="20" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2" stroke-linecap="round" stroke-linejoin="round">
                    <circle cx="12" cy="12" r="5"></circle>
                    <line x1="12" y1="1" x2="12" y2="3"></line>
                    <line x1="12" y1="21" x2="12" y2="23"></line>
                    <line x1="4.22" y1="4.22" x2="5.64" y2="5.64"></line>
                    <line x1="18.36" y1="18.36" x2="19.78" y2="19.78"></line>
                    <line x1="1" y1="12" x2="3" y2="12"></line>
                    <line x1="21" y1="12" x2="23" y2="12"></line>
                    <line x1="4.22" y1="19.78" x2="5.64" y2="18.36"></line>
                    <line x1="18.36" y1="5.64" x2="19.78" y2="4.22"></line>
                </svg>`;
    } else {
      toggleThemeBtn.innerHTML = `
                <svg xmlns="http://www.w3.org/2000/svg" width="20" height="20" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2" stroke-linecap="round" stroke-linejoin="round">
                    <path d="M21 12.79A9 9 0 1 1 11.21 3 7 7 0 0 0 21 12.79z"></path>
                </svg>`;
    }
  }

  scrollToBottom();
})();
