(function() {
    const usernameInput = document.getElementById('username');
    const passwordInput = document.getElementById('password');
    const togglePasswordBtn = document.getElementById('toggle-password');
    const loginBtn = document.getElementById('login-btn');
    const googleLoginBtn = document.getElementById('google-login-btn');
    const toggleThemeBtn = document.getElementById('toggle-theme');
    const rememberCheckbox = document.getElementById('remember');
    const registerLink = document.getElementById('register-link');
    const forgotPasswordLink = document.querySelector('.forgot-password');
    const authMessageEl = document.getElementById('auth-message');
    
    let isLoginMode = true; // true = login, false = register

    // Storage utility
    const storage = {
        save: function(key, value) {
            try {
                localStorage.setItem(key, JSON.stringify(value));
                return true;
            } catch(e) {
                console.error('Error saving to localStorage:', e);
                return false;
            }
        },
        load: function(key) {
            try {
                const item = localStorage.getItem(key);
                return item ? JSON.parse(item) : null;
            } catch(e) {
                console.error('Error loading from localStorage:', e);
                return null;
            }
        }
    };
    
    // Exibir mensagem de sucesso ou erro
    function showMessage(message, isError = false) {
        authMessageEl.textContent = message;
        authMessageEl.className = 'auth-message ' + (isError ? 'error' : 'success');
        
        // Esconder a mensagem após 5 segundos se for de sucesso
        if (!isError) {
            setTimeout(() => {
                authMessageEl.className = 'auth-message';
            }, 5000);
        }
    }
    
    // Limpar mensagens
    function clearMessages() {
        authMessageEl.className = 'auth-message';
    }

    // Load theme preference
    function loadTheme() {
        const savedTheme = storage.load('theme') || 'light';
        document.body.setAttribute('data-theme', savedTheme);
        updateThemeIcon(savedTheme);
    }

    // Toggle between light and dark theme
    function toggleTheme() {
        const isDark = document.body.getAttribute('data-theme') === 'dark';
        const newTheme = isDark ? 'light' : 'dark';
        document.body.setAttribute('data-theme', newTheme);
        storage.save('theme', newTheme);
        updateThemeIcon(newTheme);
    }

    // Update theme icon based on current theme
    function updateThemeIcon(theme) {
        if (theme === 'dark') {
            toggleThemeBtn.innerHTML = `
                <svg xmlns="http://www.w3.org/2000/svg" width="20" height="20" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2" stroke-linecap="round" stroke-linejoin="round">
                    <circle cx="12" cy="12" r="5"></circle>
                    <line x1="12" y1="1" x2="12" y2="3"></line>
                    <line x1="12" y1="21" x2="12" y2="23"></line>
                    <line x1="4.22" y1="4.22" x2="5.64" y2="5.64"></line>
                    <line x1="18.36" y1="18.36" x2="19.78" y2="19.78"></line>
                    <line x1="1" y1="12" x2="3" y2="12"></line>
                    <line x1="21" y1="12" x2="23" y2="12"></line>
                    <line x1="4.22" y1="19.78" x2="5.64" y2="18.36"></line>
                    <line x1="18.36" y1="5.64" x2="19.78" y2="4.22"></line>
                </svg>`;
        } else {
            toggleThemeBtn.innerHTML = `
                <svg xmlns="http://www.w3.org/2000/svg" width="20" height="20" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2" stroke-linecap="round" stroke-linejoin="round">
                    <path d="M21 12.79A9 9 0 1 1 11.21 3 7 7 0 0 0 21 12.79z"></path>
                </svg>`;
        }
    }

    // Toggle password visibility
    function togglePasswordVisibility() {
        const type = passwordInput.getAttribute('type') === 'password' ? 'text' : 'password';
        passwordInput.setAttribute('type', type);
        
        // Update icon based on visibility state
        if (type === 'text') {
            togglePasswordBtn.innerHTML = `
                <svg xmlns="http://www.w3.org/2000/svg" width="20" height="20" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2" stroke-linecap="round" stroke-linejoin="round">
                    <path d="M17.94 17.94A10.07 10.07 0 0 1 12 20c-7 0-11-8-11-8a18.45 18.45 0 0 1 5.06-5.94M9.9 4.24A9.12 9.12 0 0 1 12 4c7 0 11 8 11 8a18.5 18.5 0 0 1-2.16 3.19m-6.72-1.07a3 3 0 1 1-4.24-4.24"></path>
                    <line x1="1" y1="1" x2="23" y2="23"></line>
                </svg>`;
        } else {
            togglePasswordBtn.innerHTML = `
                <svg xmlns="http://www.w3.org/2000/svg" width="20" height="20" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2" stroke-linecap="round" stroke-linejoin="round">
                    <path d="M1 12s4-8 11-8 11 8 11 8-4 8-11 8-11-8-11-8z"></path>
                    <circle cx="12" cy="12" r="3"></circle>
                </svg>`;
        }
    }

    // Try to load saved username if "remember me" was checked
    function loadSavedCredentials() {
        const savedUser = storage.load('rememberedUser');
        if (savedUser) {
            usernameInput.value = savedUser;
            rememberCheckbox.checked = true;
        }
    }

    // Alternar entre login e registro
    function toggleAuthMode() {
        isLoginMode = !isLoginMode;
        
        // Atualizar texto do botão
        loginBtn.textContent = isLoginMode ? 'Entrar' : 'Registrar';
        
        // Atualizar texto do link
        registerLink.textContent = isLoginMode ? 'Registre-se' : 'Fazer login';
        
        // Atualizar título
        document.querySelector('.login-header p').textContent = 
            isLoginMode ? 'Faça login para continuar' : 'Crie sua conta';
        
        // Mostrar/ocultar dica de recuperação de senha
        const passwordRecoveryInfo = document.querySelector('.password-recovery-info');
        if (passwordRecoveryInfo) {
            passwordRecoveryInfo.style.display = isLoginMode ? 'block' : 'none';
        }
        
        clearMessages();
    }
    
    // Função para validar email
    function isValidEmail(email) {
        const emailRegex = /^[^\s@]+@[^\s@]+\.[^\s@]+$/;
        return emailRegex.test(email);
    }
    
    // Validar campos antes de submeter
    function validateFields() {
        const email = usernameInput.value.trim();
        const password = passwordInput.value.trim();
        
        if (!email || !password) {
            showMessage('Por favor, preencha todos os campos', true);
            return false;
        }
        
        if (!isValidEmail(email)) {
            showMessage('Por favor, insira um e-mail válido', true);
            return false;
        }
        
        if (password.length < 6) {
            showMessage('A senha deve ter pelo menos 6 caracteres', true);
            return false;
        }
        
        return true;
    }
    
    // Handle login submit
    async function handleLogin() {
        if (!validateFields()) return;
        
        const email = usernameInput.value.trim();
        const password = passwordInput.value.trim();
        
        // Desabilitar botão durante a autenticação
        loginBtn.disabled = true;
        loginBtn.textContent = isLoginMode ? 'Entrando...' : 'Registrando...';
        
        try {
            let result;
            
            if (isLoginMode) {
                // Login com o Supabase
                result = await loginWithEmail(email, password);
                
                if (result.success) {
                    // Save email if "remember me" is checked
                    if (rememberCheckbox.checked) {
                        storage.save('rememberedUser', email);
                    } else {
                        localStorage.removeItem('rememberedUser');
                    }
                    
                    // Redirecionar para a página de destino
                    window.location.href = getRedirectURL();
                } else {
                    showMessage(result.error, true);
                }
            } else {
                // Registro com o Supabase
                result = await registerWithEmail(email, password);
                
                if (result.success) {
                    showMessage(result.message);
                    
                    // Se o registro for bem-sucedido e não precisar de verificação de e-mail,
                    // redirecionar para a página de chat
                    if (result.user && result.user.confirmed_at) {
                        setTimeout(() => {
                            window.location.href = getRedirectURL();
                        }, 1500);
                    } else {
                        // Voltar para o modo de login após 2 segundos
                        setTimeout(() => {
                            isLoginMode = true;
                            loginBtn.textContent = 'Entrar';
                            document.querySelector('.login-header p').textContent = 'Faça login para continuar';
                            registerLink.textContent = 'Registre-se';
                        }, 2000);
                    }
                } else {
                    showMessage(result.error, true);
                }
            }
        } catch (error) {
            showMessage('Erro durante a autenticação. Tente novamente.', true);
            console.error('Erro de autenticação:', error);
        } finally {
            // Reabilitar botão
            loginBtn.disabled = false;
            loginBtn.textContent = isLoginMode ? 'Entrar' : 'Registrar';
        }
    }
    
    // Função para lidar com a recuperação de senha
    async function handlePasswordReset(e) {
        e.preventDefault();
        
        const email = usernameInput.value.trim();
        
        if (!email) {
            showMessage('Digite seu e-mail para recuperar a senha', true);
            return;
        }
        
        if (!isValidEmail(email)) {
            showMessage('Por favor, insira um e-mail válido', true);
            return;
        }
        
        // Desabilitar link durante o processo e mostrar feedback
        const forgotBtn = document.querySelector('.forgot-password');
        const originalText = forgotBtn.textContent;
        forgotBtn.textContent = 'Enviando...';
        forgotBtn.style.opacity = '0.7';
        forgotBtn.style.pointerEvents = 'none';
        
        try {
            const result = await resetPassword(email);
            
            if (result.success) {
                showMessage(result.message);
                // Manter o botão desabilitado, já que o email foi enviado
                forgotBtn.textContent = 'Email enviado';
                setTimeout(() => {
                    forgotBtn.textContent = originalText;
                    forgotBtn.style.opacity = '1';
                    forgotBtn.style.pointerEvents = 'auto';
                }, 5000);
            } else {
                showMessage(result.error, true);
                // Restaurar o botão
                forgotBtn.textContent = originalText;
                forgotBtn.style.opacity = '1';
                forgotBtn.style.pointerEvents = 'auto';
            }
        } catch (error) {
            showMessage('Erro ao solicitar recuperação de senha', true);
            console.error('Erro de recuperação de senha:', error);
            // Restaurar o botão
            forgotBtn.textContent = originalText;
            forgotBtn.style.opacity = '1';
            forgotBtn.style.pointerEvents = 'auto';
        }
    }

    // Verificar se já existe um usuário logado
    async function checkExistingSession() {
        try {
            const user = await checkSession();
            if (user) {
                // Já está logado, redirecionar para o chat
                window.location.href = 'index.html';
            }
        } catch (error) {
            console.error('Erro ao verificar sessão:', error);
        }
    }

    // Event listeners
    togglePasswordBtn.addEventListener('click', togglePasswordVisibility);
    toggleThemeBtn.addEventListener('click', toggleTheme);
    loginBtn.addEventListener('click', handleLogin);
    googleLoginBtn.addEventListener('click', handleGoogleLogin);
    registerLink.addEventListener('click', function(e) {
        e.preventDefault();
        toggleAuthMode();
    });
    forgotPasswordLink.addEventListener('click', handlePasswordReset);
    
    // Handle form submission on Enter key
    document.addEventListener('keydown', function(e) {
        if (e.key === 'Enter') {
            handleLogin();
        }
    });

    // Função para lidar com o login do Google
    async function handleGoogleLogin() {
        try {
            showMessage("Redirecionando para o login com Google...");
            const result = await loginWithGoogle();
            if (!result.success) {
                showMessage(result.error, true);
            }
        } catch (error) {
            console.error("Erro ao iniciar login com Google:", error);
            showMessage("Erro ao iniciar login com Google. Tente novamente.", true);
        }
    }
    
    // Verificar se há uma resposta de autenticação OAuth
    async function checkOAuthResponse() {
        // Verificar se há um hash na URL (indicando redirecionamento OAuth)
        if (window.location.hash || window.location.search.includes('access_token')) {
            showMessage("Processando autenticação...");
            try {
                const result = await handleOAuthResponse();
                if (result.success) {
                    showMessage("Login com Google bem-sucedido! Redirecionando...");
                    setTimeout(() => {
                        window.location.href = getRedirectURL();
                    }, 1000);
                } else {
                    showMessage(result.error, true);
                }
            } catch (error) {
                console.error("Erro ao processar resposta OAuth:", error);
                showMessage("Erro ao finalizar autenticação.", true);
            }
        }
    }
    
    // Função para obter a URL de redirecionamento da query string
    function getRedirectURL() {
        const urlParams = new URLSearchParams(window.location.search);
        return urlParams.get('redirect') || 'index.html';
    }
    
    // Initialize
    loadTheme();
    loadSavedCredentials();
    checkExistingSession();
    checkOAuthResponse();
    
    // Esconder dica de recuperação de senha no modo de registro
    if (!isLoginMode) {
        const passwordRecoveryInfo = document.querySelector('.password-recovery-info');
        if (passwordRecoveryInfo) {
            passwordRecoveryInfo.style.display = 'none';
        }
    }
})();