// js/users.js
import { supabase } from './supabase.js';

// Função para inicializar a página de usuários
async function initUsersPage() {
    const userSearchInput = document.getElementById('user-search');
    const addUserBtn = document.getElementById('add-user-btn');
    const userCardsContainer = document.getElementById('user-cards-container');
    const editUserPopupOverlay = document.getElementById('edit-user-popup-overlay');
    const editUserForm = document.getElementById('edit-user-form');
    const cancelEditBtn = document.getElementById('cancel-edit-btn');
    const editMessage = document.getElementById('edit-message');

    let allUsers = []; // Para armazenar todos os usuários e facilitar a busca local

    // Função para carregar usuários do Supabase
    async function fetchUsers() {
        userCardsContainer.innerHTML = '<div class="loading-spinner" style="display: block;"></div>'; // Mostrar spinner
        try {
            const { data, error } = await supabase
                .from('usuario')
                .select('*')
                .order('username', { ascending: true }); // Ordena por nome de usuário

            if (error) throw error;
            allUsers = data;
            renderUserCards(allUsers);
        } catch (error) {
            console.error('Erro ao buscar usuários:', error.message);
            userCardsContainer.innerHTML = `<div class="content-placeholder"><h2>Erro</h2><p>Não foi possível carregar os usuários: ${error.message}</p></div>`;
        } finally {
            userCardsContainer.querySelector('.loading-spinner').style.display = 'none'; // Esconder spinner
        }
    }

    // Função para renderizar os cards de usuários
    function renderUserCards(usersToRender) {
        userCardsContainer.innerHTML = ''; // Limpa os cards existentes
        if (usersToRender.length === 0) {
            userCardsContainer.innerHTML = '<div class="content-placeholder"><h2>Nenhum usuário encontrado</h2><p>Tente ajustar sua busca ou adicione um novo usuário.</p></div>';
            return;
        }
        usersToRender.forEach(user => {
            const card = document.createElement('div');
            card.className = 'user-card';
            const statusText = user.ativo ? 'Ativo' : 'Inativo';
            const statusClass = user.ativo ? 'status-active' : 'status-inactive';

            card.innerHTML = `
                <div class="user-card-header">
                    <h3>${user.username}</h3>
                    <i class="fas fa-edit edit-icon" data-user-id="${user.id}"></i>
                </div>
                <div class="user-card-info">
                    <p>Perfil: <span>${user.perfil}</span></p>
                    <div class="user-status">
                        Status: <span class="status-indicator ${statusClass}"></span> ${statusText}
                    </div>
                </div>
            `;
            userCardsContainer.appendChild(card);
        });

        // Adicionar listeners para os botões de edição
        document.querySelectorAll('.edit-icon').forEach(icon => {
            icon.addEventListener('click', (e) => {
                const userId = e.currentTarget.dataset.userId;
                openEditUserPopup(userId);
            });
        });
    }

    // Busca instantânea
    userSearchInput.addEventListener('input', (e) => {
        const searchTerm = e.target.value.toLowerCase();
        const filteredUsers = allUsers.filter(user =>
            user.username.toLowerCase().includes(searchTerm) ||
            user.perfil.toLowerCase().includes(searchTerm)
        );
        renderUserCards(filteredUsers);
    });

    // Botão "Novo Usuário" - Usa a função global loadPage do main.js
    addUserBtn.addEventListener('click', () => {
        if (typeof window.loadPage === 'function') {
            window.loadPage('register');
        } else {
            console.error("Função 'loadPage' não disponível. Verifique main.js.");
        }
    });

    // Abrir popup de edição de usuário
    async function openEditUserPopup(userId) {
        editMessage.textContent = ''; // Limpa mensagens anteriores
        try {
            const { data, error } = await supabase
                .from('usuario')
                .select('*')
                .eq('id', userId)
                .single();

            if (error) throw error;

            document.getElementById('edit-user-id').value = data.id;
            document.getElementById('edit-username').value = data.username;
            document.getElementById('edit-password').value = ''; // Senha sempre vazia para não alterar
            document.getElementById('edit-profile').value = data.perfil;
            document.getElementById('edit-active').checked = data.ativo;

            editUserPopupOverlay.classList.add('active');
        } catch (error) {
            console.error('Erro ao carregar dados do usuário para edição:', error.message);
            alert(`Erro ao carregar dados do usuário: ${error.message}`);
        }
    }

    // Salvar edição de usuário
    editUserForm.addEventListener('submit', async (e) => {
        e.preventDefault();

        const userId = document.getElementById('edit-user-id').value;
        const username = document.getElementById('edit-username').value;
        const password = document.getElementById('edit-password').value;
        const profile = document.getElementById('edit-profile').value;
        const active = document.getElementById('edit-active').checked;

        if (!username || !profile) {
            editMessage.textContent = 'Por favor, preencha os campos obrigatórios (Usuário e Perfil).';
            editMessage.style.color = 'var(--error-color)';
            return;
        }

        try {
            const updateData = { username: username, perfil: profile, ativo: active };
            if (password) { // Apenas atualiza a senha se um novo valor for fornecido
                updateData.senha = password;
            }

            const { error } = await supabase
                .from('usuario')
                .update(updateData)
                .eq('id', userId);

            if (error) throw error;

            editMessage.textContent = 'Usuário atualizado com sucesso!';
            editMessage.style.color = 'var(--success-color)';
            await fetchUsers(); // Recarrega a lista de usuários para mostrar as alterações
            setTimeout(() => {
                editUserPopupOverlay.classList.remove('active');
            }, 1000); // Fecha o popup após 1 segundo
        } catch (error) {
            console.error('Erro ao atualizar usuário:', error.message);
            editMessage.textContent = `Erro ao atualizar usuário: ${error.message}`;
            editMessage.style.color = 'var(--error-color)';
        }
    });

    // Cancelar edição
    cancelEditBtn.addEventListener('click', () => {
        editUserPopupOverlay.classList.remove('active');
    });

    // Fechar popup clicando fora
    editUserPopupOverlay.addEventListener('click', (e) => {
        if (e.target === editUserPopupOverlay) {
            editUserPopupOverlay.classList.remove('active');
        }
    });

    // Carregar usuários ao iniciar a página
    fetchUsers();
}

// Expõe a função para ser chamada pelo main.js quando a página for carregada via AJAX
window.initUsersPage = initUsersPage;
