// js/main.js
import { supabase } from './supabase.js'; // Importa a instância do Supabase

document.addEventListener('DOMContentLoaded', () => {
    const sidebar = document.getElementById('sidebar');
    const menuToggle = document.getElementById('menu-toggle');
    const pageTitle = document.getElementById('page-title');
    const mainContent = document.getElementById('main-content');
    const logoutBtn = document.getElementById('logout-btn');
    const themeSwitchBtn = document.getElementById('theme-switch-btn');
    const globalSettingsBtn = document.getElementById('global-settings-btn');
    const settingsPopupOverlay = document.getElementById('settings-popup-overlay');
    const saveSettingsBtn = document.getElementById('save-settings-btn');
    const cancelSettingsBtn = document.getElementById('cancel-settings-btn');
    const webhookUrlInput = document.getElementById('webhook-url');
    const navLinks = document.querySelectorAll('.sidebar-nav a');
    const topbar = document.querySelector('.topbar');

    // Carregar tema salvo no localStorage
    const savedTheme = localStorage.getItem('theme') || 'light-theme';
    document.body.className = savedTheme;
    updateThemeSwitchIcon();

    // Carregar webhook salvo no localStorage (ou usar o padrão)
    const savedWebhook = localStorage.getItem('chatWebhookUrl') || 'https://n8n.ichatbot.com.br/webhook/chat-com-ia';
    webhookUrlInput.value = savedWebhook;

    // Função para alternar o tema e o ícone
    function toggleTheme() {
        if (document.body.classList.contains('light-theme')) {
            document.body.classList.remove('light-theme');
            document.body.classList.add('dark-theme');
            localStorage.setItem('theme', 'dark-theme');
        } else {
            document.body.classList.remove('dark-theme');
            document.body.classList.add('light-theme');
            localStorage.setItem('theme', 'light-theme');
        }
        updateThemeSwitchIcon();
    }

    // Atualiza o ícone do botão de tema
    function updateThemeSwitchIcon() {
        const themeIcon = themeSwitchBtn.querySelector('i');
        if (document.body.classList.contains('light-theme')) {
            themeIcon.className = 'fas fa-sun';
        } else {
            themeIcon.className = 'fas fa-moon';
        }
    }

    // Toggle do menu lateral
    menuToggle.addEventListener('click', () => {
        sidebar.classList.toggle('collapsed');
        topbar.classList.toggle('sidebar-expanded');
        // Em telas menores, o sidebar deve ser um overlay
        if (window.innerWidth <= 768) {
            sidebar.classList.toggle('expanded');
        }
    });

    // Evento de clique para o botão de tema no rodapé
    themeSwitchBtn.addEventListener('click', toggleTheme);

    // Logout
    logoutBtn.addEventListener('click', (e) => {
        e.preventDefault();
        if (confirm('Tem certeza que deseja sair?')) {
            sessionStorage.removeItem('loggedInUser'); // Limpa a sessão
            window.location.href = 'index.html'; // Redireciona para a tela de login
        }
    });

    // Abrir popup de configurações globais
    globalSettingsBtn.addEventListener('click', () => {
        settingsPopupOverlay.classList.add('active');
        webhookUrlInput.value = localStorage.getItem('chatWebhookUrl') || 'https://n8n.ichatbot.com.br/webhook/chat-com-ia';
    });

    // Salvar configurações
    saveSettingsBtn.addEventListener('click', () => {
        const newWebhookUrl = webhookUrlInput.value;
        if (newWebhookUrl) {
            localStorage.setItem('chatWebhookUrl', newWebhookUrl);
            alert('Configurações salvas com sucesso!');
            settingsPopupOverlay.classList.remove('active');
        } else {
            alert('A URL do Webhook não pode estar vazia.');
        }
    });

    // Cancelar configurações
    cancelSettingsBtn.addEventListener('click', () => {
        settingsPopupOverlay.classList.remove('active');
    });

    // Fechar popup clicando fora
    settingsPopupOverlay.addEventListener('click', (e) => {
        if (e.target === settingsPopupOverlay) {
            settingsPopupOverlay.classList.remove('active');
        }
    });

    // Função para carregar conteúdo da página via AJAX
    // Exporta para ser acessível globalmente (ex: de users.js para carregar register.html)
    window.loadPage = async function(pageName) {
        let pageUrl = '';
        let title = '';

        // Remove a classe 'active' de todos os links do menu
        navLinks.forEach(link => link.classList.remove('active'));

        switch (pageName) {
            case 'dashboard':
                pageUrl = 'dashboard.html';
                title = 'Dashboard';
                break;
            case 'users':
                pageUrl = 'users.html';
                title = 'Gerenciamento de Usuários';
                break;
            case 'chat':
                pageUrl = 'chat.html';
                title = 'Chat com IA';
                break;
            case 'reports':
                pageUrl = 'reports.html';
                title = 'Relatórios';
                break;
            case 'register': // Adicionado para carregar a página de cadastro de usuário
                pageUrl = 'register.html';
                title = 'Cadastro de Usuário';
                break;
            case 'settings-placeholder': // Este é um placeholder para o botão, a ação real é o popup
                title = 'Configurações';
                // Não retorna URL para não carregar HTML, a ação é via popup
                break;
            default:
                pageUrl = 'dashboard.html';
                title = 'Dashboard';
        }

        pageTitle.textContent = title;

        // Adiciona a classe 'active' ao link clicado
        const activeLink = document.querySelector(`.sidebar-nav a[data-page="${pageName}"]`);
        if (activeLink) {
            activeLink.classList.add('active');
        }

        if (pageUrl && pageUrl !== 'settings-placeholder') { // Não tenta carregar HTML para o placeholder
            try {
                const response = await fetch(pageUrl);
                if (!response.ok) {
                    throw new Error(`HTTP error! status: ${response.status}`);
                }
                const html = await response.text();
                mainContent.innerHTML = html;

                // Executa scripts dentro do HTML carregado
                // Isso é importante para scripts que não são módulos (type="module")
                mainContent.querySelectorAll('script').forEach(oldScript => {
                    const newScript = document.createElement('script');
                    // Copia atributos (src, type, etc.)
                    Array.from(oldScript.attributes).forEach(attr => newScript.setAttribute(attr.name, attr.value));
                    newScript.appendChild(document.createTextNode(oldScript.innerHTML));
                    oldScript.parentNode.replaceChild(newScript, oldScript);
                });

                // Inicializa scripts específicos da página se existirem
                // A função global `initXPage` deve ser exposta pelos respectivos scripts (ex: `window.initChatPage = initChatPage;`)
                if (pageName === 'chat' && typeof window.initChatPage === 'function') {
                    window.initChatPage();
                } else if (pageName === 'users' && typeof window.initUsersPage === 'function') {
                    window.initUsersPage();
                } else if (pageName === 'register' && typeof window.initRegisterPage === 'function') {
                    window.initRegisterPage();
                }

            } catch (error) {
                console.error(`Não foi possível carregar a página ${pageName}:`, error);
                mainContent.innerHTML = `<div class="content-placeholder"><h2>Erro ao carregar página</h2><p>Não foi possível carregar o conteúdo de ${title}.</p></div>`;
            }
        } else if (pageName === 'dashboard') {
             // Conteúdo padrão para dashboard
             mainContent.innerHTML = `
                <div class="content-placeholder">
                    <h2>Bem-vindo!</h2>
                    <p>Navegue pelo menu lateral para acessar as funcionalidades do sistema.</p>
                </div>
            `;
        }
    };

    // Event listeners para navegação no menu
    navLinks.forEach(link => {
        link.addEventListener('click', (e) => {
            e.preventDefault();
            const page = e.currentTarget.dataset.page;
            window.loadPage(page);
            // Em telas pequenas, colapsar o menu após a seleção
            if (window.innerWidth <= 768) {
                sidebar.classList.remove('expanded');
            }
        });
    });

    // Carregar a página inicial (dashboard) ao carregar main.html
    window.loadPage('dashboard');

    // Verifica se o usuário está logado ao carregar main.html
    const loggedInUser = sessionStorage.getItem('loggedInUser');
    if (!loggedInUser) {
        window.location.href = 'index.html'; // Redireciona se não estiver logado
    }
});
