// js/chat.js

// Expõe esta função globalmente para que main.js possa chamá-la
window.initChatPage = initChatPage;

async function initChatPage() {
    const chatMessages = document.getElementById('chat-messages');
    const chatMessageInput = document.getElementById('chat-message-input');
    const sendChatBtn = document.getElementById('send-chat-btn');
    const clearChatBtn = document.getElementById('clear-chat-btn');
    const confirmClearChatPopup = document.getElementById('clear-chat-confirm-popup');
    const confirmClearChatBtn = document.getElementById('confirm-clear-chat-btn');
    const cancelClearChatBtn = document.getElementById('cancel-clear-chat-btn');

    let chatHistory = JSON.parse(sessionStorage.getItem('chatHistory')) || [];

    // Função para obter a URL do webhook
    function getWebhookUrl() {
        return localStorage.getItem('chatWebhookUrl') || 'https://n8n.ichatbot.com.br/webhook/chat-com-ia';
    }

    // Função para formatar a hora (HH:mm estática)
    function formatTime(date) {
        return date.toTimeString().slice(0, 5); // HH:mm
    }

    // Renderiza uma mensagem no chat
    function appendMessage(message, sender, timestamp) {
        const messageBubble = document.createElement('div');
        messageBubble.classList.add('message-bubble', sender);

        const avatar = document.createElement('img');
        avatar.classList.add('message-avatar');
        avatar.src = `assets/${sender}-icon.png`; // Usa user-icon.png ou bot-icon.png
        avatar.alt = sender === 'user' ? 'Você' : 'Bot';

        const messageContent = document.createElement('div');
        messageContent.classList.add('message-content');

        const text = document.createElement('p');
        text.textContent = message;

        const time = document.createElement('span');
        time.classList.add('message-time');
        time.textContent = timestamp;

        messageContent.appendChild(text);
        messageContent.appendChild(time);

        if (sender === 'user') {
            messageBubble.appendChild(messageContent);
            messageBubble.appendChild(avatar);
        } else {
            messageBubble.appendChild(avatar);
            messageBubble.appendChild(messageContent);
        }

        chatMessages.appendChild(messageBubble);
        // Scroll para o final da conversa
        chatMessages.scrollTop = chatMessages.scrollHeight;
    }

    // Carrega o histórico de mensagens da sessão
    function loadChatHistory() {
        chatMessages.innerHTML = ''; // Limpa antes de carregar
        chatHistory.forEach(msg => {
            appendMessage(msg.message, msg.sender, msg.timestamp);
        });
    }

    // Envia a mensagem para o webhook do bot
    async function sendMessageToBot(message) {
        const webhookUrl = getWebhookUrl();
        if (!webhookUrl) {
            appendMessage("Erro: URL do webhook não configurada. Vá em Configurações.", "bot", formatTime(new Date()));
            return;
        }

        try {
            const response = await fetch(webhookUrl, {
                method: 'POST',
                headers: {
                    'Content-Type': 'application/json',
                },
                body: JSON.stringify({ message: message }),
            });

            // Adicionado tratamento para status 204 (No Content)
            if (response.status === 204) {
                console.warn('Resposta do bot com status 204: Sem conteúdo para o chat.');
                appendMessage("O bot processou sua mensagem, mas não enviou uma resposta de chat.", "bot", formatTime(new Date()));
                return;
            }

            if (!response.ok) {
                const errorText = await response.text(); // Tenta ler o corpo da resposta de erro
                console.error('Erro HTTP do bot:', response.status, response.statusText, errorText);
                throw new Error(`Erro do servidor/bot: ${response.status} ${response.statusText}`);
            }

            // Leia a resposta como texto primeiro para depuração robusta
            const responseText = await response.text();
            console.log('Resposta bruta do webhook:', responseText);

            if (responseText.trim() === '') {
                console.warn('Resposta do bot com status 200, mas corpo vazio ou branco.');
                appendMessage("O bot processou sua mensagem, mas não enviou uma resposta de chat (corpo vazio).", "bot", formatTime(new Date()));
                return;
            }

            let data;
            try {
                data = JSON.parse(responseText); // Tenta parsear o texto como JSON
            } catch (parseError) {
                console.error('Erro ao parsear JSON da resposta:', parseError);
                console.error('Conteúdo que tentou ser parseado:', responseText);
                appendMessage(`Erro na resposta do bot: Formato inválido. Detalhes: ${parseError.message}`, "bot", formatTime(new Date()));
                return;
            }

            const botMessage = data.response || data.message || "Desculpe, não consegui processar sua solicitação."; // Tenta 'response' ou 'message'
            const timestamp = formatTime(new Date());

            appendMessage(botMessage, 'bot', timestamp);
            chatHistory.push({ message: botMessage, sender: 'bot', timestamp: timestamp });
            sessionStorage.setItem('chatHistory', JSON.stringify(chatHistory));

        } catch (error) {
            console.error('Falha ao enviar mensagem ou processar resposta do bot:', error);
            // Mensagem de erro mais detalhada para o usuário
            appendMessage(`Erro ao conectar com o bot: ${error.message}. Por favor, verifique a URL do webhook nas configurações ou tente novamente mais tarde.`, 'bot', formatTime(new Date()));
        }
    }

    // Evento de envio de mensagem ao clicar no botão
    sendChatBtn.addEventListener('click', async () => {
        const message = chatMessageInput.value.trim();
        if (message) {
            const timestamp = formatTime(new Date());
            appendMessage(message, 'user', timestamp);
            chatHistory.push({ message: message, sender: 'user', timestamp: timestamp });
            sessionStorage.setItem('chatHistory', JSON.stringify(chatHistory));

            chatMessageInput.value = ''; // Limpa o input
            chatMessageInput.style.height = 'auto'; // Reseta a altura do textarea

            await sendMessageToBot(message);
        }
    });

    // Enviar mensagem com Enter (Shift + Enter para nova linha)
    chatMessageInput.addEventListener('keypress', (e) => {
        if (e.key === 'Enter' && !e.shiftKey) {
            e.preventDefault(); // Evita a quebra de linha padrão do Enter
            sendChatBtn.click(); // Dispara o clique do botão de envio
        }
    });

    // Ajustar altura do textarea dinamicamente
    chatMessageInput.addEventListener('input', () => {
        chatMessageInput.style.height = 'auto'; // Reseta a altura para calcular corretamente
        chatMessageInput.style.height = chatMessageInput.scrollHeight + 'px'; // Ajusta para o conteúdo
    });

    // Botão de limpar histórico - Abre popup de confirmação
    clearChatBtn.addEventListener('click', () => {
        confirmClearChatPopup.classList.add('active');
    });

    // Confirmação para limpar histórico
    confirmClearChatBtn.addEventListener('click', () => {
        chatHistory = []; // Limpa o array do histórico
        sessionStorage.removeItem('chatHistory'); // Remove do sessionStorage
        chatMessages.innerHTML = ''; // Limpa a interface do chat
        confirmClearChatPopup.classList.remove('active'); // Fecha o popup
        appendMessage("Histórico de chat limpo.", "bot", formatTime(new Date())); // Adiciona uma mensagem de confirmação no chat
    });

    // Cancelar a limpeza do histórico
    cancelClearChatBtn.addEventListener('click', () => {
        confirmClearChatPopup.classList.remove('active');
    });

    // Fechar popup clicando fora da área de conteúdo
    confirmClearChatPopup.addEventListener('click', (e) => {
        if (e.target === confirmClearChatPopup) {
            confirmClearChatPopup.classList.remove('active');
        }
    });

    // Carregar histórico ao iniciar a página de chat
    loadChatHistory();
}
